#----- Loading packages -----#
library(plyr) #Version 1.8.6
library(data.table) #Version 1.13.4
library(scales) #Version 1.1.1
library(maps) #Version 3.3.0
library(tidyverse) #Version 1.3.0
library(betareg) #Version 3.1.3
library(mgcv) #Version 1.8.33
library(raster) #Version 3.4.5
library(sf) #Version 0.9.7
library(ggspatial) #Version 1.1.5
library(rgeos) #Version 0.5.5
library(rnaturalearth) #Version 0.1.0


#----- Setting WD -----#
wd <- 'XXX' #Insert your working directory here
setwd(wd)


#----- Sourcing functions -----#
source(paste0(wd,'/Scripts/Siqueira_etal_RFunctions.R'))


#----- Loading data -----#
pa_rich <- read.csv(paste0(wd,'/Data/Richness_PerCell_PresenceAbsence.csv'), header=T, sep = ",", stringsAsFactors = T)

pa_prop <- read.csv(paste0(wd,'/Data/Proportion_PerCell_PresenceAbsence.csv'), header=T, sep = ",", stringsAsFactors = T)

size_df <- read.csv(paste0(wd,'/Data/BodySize_PerCell_PresenceAbsence.csv'), header=T, sep = ",", stringsAsFactors = T)


#----- Map adjustments -----#
center <- 160

poly <- cbind(pa_rich$Long, pa_rich$Lat)
polygons <- fun_points(poly)

pa_rich$polyLong <- polygons[,1]
pa_rich$polyLat <- polygons[,2]

ax_coord <- seq(-150,150,50)
ax_lab <- sapply(ax_coord,function(x){
  x <- x - center
  x <- ifelse(x<(-180),x+360,x)
  if(sum(diff(x)>300,na.rm=T) >0){
    id <- x < 0
    x <- rbind(x[id,],c(NA,NA),x[!id,])
  }
  as.character(x)
})

cols <-  colorRampPalette(c('#3B9AB2', '#78B7C5', '#EBCC2A', '#E1AF00','#F21A00'))(max(pa_rich[,2:7]))


#----- Figure 1 -----#
par(mfrow = c(3,2),oma = c(0,2,0,0), mar = c(0,0,0,0), bg = 'white')
for (i in 2:7){
  df <- pa_rich[,c('polyLong','polyLat')]
  cols_df <- vector(); for(x in 1:length(pa_rich[,i])) {cols_df[x] <- cols[pa_rich[,i][x]]}
  df$cols <- cols_df; df$val <- pa_rich[,i]
  df <- df[complete.cases(df),];df <- df[with(df, order(df$val)),]
  plot.map('world', center=center, col='transparent',bg='white', fill=TRUE,ylim=c(-45,45),mar = c(0,2,0,2), border = "transparent")
  axis (1,pos = -46, at = ax_coord, labels = ax_lab, tcl = -0.3, mgp = c(3, 0.5, 0), cex.axis = 0.8, las = 1, lwd = 0.8)
  axis (2,pos = -180, at = seq(-45,45,15), labels = as.character(seq(-45,45,15)), tcl = -0.3, mgp = c(3, 0.5, 0), cex.axis = 0.8, las = 2, lwd = 0.8)
  points(df[,2]~df[,1], pch=16,col=df$cols,cex=0.8)
  plot.map('world', center=160, col='#CCCDCE', fill=TRUE,ylim=c(-45,45),add = T, border = 'transparent')
  segments(-180,-46,-180,46, lwd = 1.5);segments(181,-46,181,46, lwd = 1.5)
  segments(-180,-46,181,-46, lwd = 1.5);segments(-180,46,181,46, lwd = 1.5)
  title(main = colnames(pa_rich)[i], line = 1.5)
}
mtext(side = 1, text = 'Longitude', line = -2, cex = 0.8, outer = T)
mtext(side = 2, text = 'Latitude', line = 0, cex = 0.9, outer = T)

### Legend
dev.off()
legend_image <- as.raster(matrix(cols, ncol=1))
legend_image <- legend_image[nrow(legend_image):1,]
plot(c(0,1),c(1,length(cols)),type = 'n', axes = F,xlab = '', ylab = '', main = 'Species richness')
text(x=0.8, y = round(seq(1,length(cols),l=7)), labels = round(seq(1,length(cols),l=7)))
rasterImage(legend_image, 0, 1, 0.1,length(cols))


#----- Running models and generating Figure 2 -----#
## Adjusting datasets
pa_long <- pa_rich %>%  gather("trophic", "occ", GC:SI)
pa_long$trophic <- as.factor(pa_long$trophic)

size_long <- size_df %>%  gather("trophic", "size", GC:SI)
size_long$trophic <- as.factor(size_long$trophic)
pa_long$size <- size_long$size

## Species proportions
prop_long <- pa_prop %>%  gather("trophic", "prop_occ", GC:SI)
prop_long$trophic <- as.factor(prop_long$trophic)
prop_long$size <- size_long$size

## Colour palette per trophic group
colours <- setNames(c('#C74221','#8BA370','#FFC857','#C6AE80','#084C61','#32221F'), 
                    c('GC','HD','MI','OM','PK','SI'))

layout( matrix(c(1,1,1,8,1,1,2,4,6,3,5,7), 3, 4, byrow = F), heights = c(1,1,1), widths = c(1,1,1,1))

### Plot1
par(mar = c(4,4,2,2))
plot(1, type="n", xlab="", ylab="", xlim=c(0, 20000), ylim=c(0, 300), tcl = -0.3, mgp = c(3, 0.5, 0), 
     las = 1, yaxt = 'n', xaxt = 'n', bty = "n", lwd = 1.7); box(lwd = 1.4)
axis (1, at = seq(0, 20000,5000), labels = as.character(seq(0, 20000,5000)), 
      tcl = -0.3, mgp = c(3, 0.5, 0), cex.axis = 1.3, las = 1, lwd = 1.4)
axis (2, at = seq(0, 300,50), labels = as.character(seq(0, 300,50)), tcl = -0.3, 
      mgp = c(3, 0.5, 0), cex.axis = 1.3, las = 1, lwd = 1.4)
mtext(side = 2, text = bquote('Species richness (spp 150'*km^-2*')'), line = 2.2, cex = 1)

r_sqr <- vector()
coef_ls <- list()

for (i in 1:6){
  sub <- pa_long %>% filter (trophic == levels(pa_long$trophic)[i])
  
  mod <- gam(occ ~ DistIAA * size, data = sub, family = 'nb', 
             correlation = corSpatial(form = ~ Long + Lat, type='gaussian'))
  summ <- summary(mod)
  r_sqr[levels(pa_long$trophic)[i]] <- summ$'r.sq'
  lo <- loess(size ~ DistIAA, data=sub, span=0.70)
  
  pred <- data.frame (DistIAA = seq (0, 20000, length = 200),
                      size = rep(lo$fitted[1], 200))
  pred <- cbind(pred, predict(mod, pred, se.fit=TRUE, type = 'link'))
  pred <- within(pred, {
    Occ <- family(mod)$linkinv(fit)
    LL <- family(mod)$linkinv(fit - 1.96 * se.fit)
    UL <- family(mod)$linkinv(fit + 1.96 * se.fit)
  })
  
  
  coef_ls[[levels(pa_long$trophic)[i]]] <- list(pt = summ$p.coeff[2], 
                                                u.pt = (summ$p.coeff[2] + 1.96 * summ$se[2]),
                                                l.pt = (summ$p.coeff[2] - 1.96 * summ$se[2]))
  
  polygon(c(pred$DistIAA,rev(pred$DistIAA)), c(pred$LL, rev(pred$UL)),
          col = alpha(colours[i],0.9), border = 'black', lwd = 0.5)
  lines(pred[,"Occ"] ~ pred$DistIAA, col = 'black',lwd = 1)
}

ord <- c('PK','OM','SI','MI','HD','GC')

### Plots 2,3,4,5,5,7
par(mar = c(4,4,2,2))
r_sqr_2 <- vector()
for (i in 1:6){
  plot(1, type="n", xlab="", ylab="", xlim=c(0, 20000), ylim=c(0, 0.5), tcl = -0.3, mgp = c(3, 0.5, 0), 
       las = 1, yaxt = 'n', xaxt = 'n', bty = "n", lwd = 1.2); box(lwd = 1.4)
  axis (1, at = seq(0, 20000,5000), labels = as.character(seq(0, 20000,5000)), 
        tcl = -0.3, mgp = c(3, 0.5, 0), cex.axis = 1, las = 1, lwd = 1.4)
  axis (2, at = seq(0, 0.5,0.1), labels = as.character(seq(0, 0.5,0.1)), tcl = -0.3, 
        mgp = c(3, 0.5, 0), cex.axis = 1, las = 1, lwd = 1.4)
  if(i == 3) {mtext(side = 2, text = bquote('Species proportion (prop 150'*km^-2*')'), line = 2, cex = 1)}
  
  sub <- prop_long %>% filter (trophic == ord[i])
  mod <- betareg(prop_occ ~ DistIAA * size, data = sub)
  
  r_sqr_2[i] <- summary(mod)$'pseudo.r.squared'
  lo <- loess(size ~ DistIAA, data=sub, span=0.70)
  
  pred <- data.frame (DistIAA = seq (0, 20000, length = 200),
                      size = rep(lo$fitted[1], 200))
  pred <- cbind(pred, fit = predict(mod, newdata = pred, type = 'response'), 
                q = predict(mod,newdata = pred, type = 'quantile', at = c(0.25,0.75)))
  
  points(sub$prop_occ ~ sub$DistIAA, col = alpha(colours[ord[i]],0.1),pch = 16, cex = 0.6)
  polygon(c(pred$DistIAA,rev(pred$DistIAA)), c(pred$q.q_0.25, rev(pred$q.q_0.75)),
          col = alpha(colours[ord[i]],0.7), border = 'black', lwd = 0.5)
  lines(pred[,"fit"] ~ pred$DistIAA, col = 'black',lwd = 1)
  lab <- bquote(italic(R)^2 == .(paste0('.',strsplit(as.character(sprintf('%.2f',r_sqr_2[i])), "\\.")[[1]][2])))
  text(2000,0.48,lab, cex = 1)
  
  
}
mtext(side = 1, text = "Distance from the Indo-Australian Archipelago (km)", line = -2, cex = 1,outer = T)

### Plot8
par(mar = c(4,4,3,3))
plot(1, type="n", xlab="", ylab="", xlim=c(-0.00026, -0.000089), ylim=c(0.5,6.5), tcl = -0.3, mgp = c(3, 0.5, 0), 
     las = 1, yaxt = 'n', xaxt = 'n', bty = "n", lwd = 1.7); box(lwd = 1)
axis (1, at = seq(-0.00026, -0.000089 ,0.00005), labels = sub("e.*",'',format(seq(-0.00026, -0.000089 ,0.00005), scientific = T)), 
      tcl = -0.3, mgp = c(3, 0.5, 0), cex.axis = 1, las = 1, lwd = 1)
axis (2, at = seq(1,6,1), labels = c('GC','HD','MI','SI','OM','PK'), tcl = -0.3, 
      mgp = c(3, 0.5, 0), cex.axis = 1.2, las = 1, lwd = 1)
mtext(side = 1, text = "Coefficient (*e-04)", line = 1.7, cex = 1)


for (i in 1:6){
  
  mean_group <- coef_ls[[rev(ord)[i]]]$'pt'
  upp_group <- coef_ls[[rev(ord)[i]]]$'u.pt'
  low_group <- coef_ls[[rev(ord)[i]]]$'l.pt'
  
  
  lines(c(low_group, upp_group), c(i,i), lwd = 1.5);
  lines(c(low_group, low_group),c(i-0.05,i+0.05), lwd = 1.5);lines(c(upp_group, upp_group),c(i-0.05,i+0.05), lwd = 1.5)
  points(mean_group,i, pch = 21, bg = colours[rev(ord)[i]], cex = 2, col = 'white')
  text(mean_group,i + 0.35,paste0('.',strsplit(as.character(sprintf('%.2f',r_sqr[rev(ord)[i]])), "\\.")[[1]][2]), cex = 1)
  
}


#----- Reef Life Survey data -----#

pa_rich_rls <- read.csv(paste0(wd,'/Data/Richness_PerSite_RLS.csv'), header=T, sep = ",", stringsAsFactors = T)

sizeob <- read.csv(paste0(wd,'/Data/MeanSize_PerSite_RLS.csv'), header=T, sep = ",", stringsAsFactors = T)

#----- Figure 3A -----#

### Pacific centered map
world <- rnaturalearth::ne_countries(scale = 'medium', returnclass = "sp")
box_cut <- bbox2SP(n = 90, s = -90, w = -100, e = 20, proj4string = world@proj4string)
world_crop <- gDifference(world, box_cut)

pacific_crop <- world_crop %>% 
  st_as_sf() %>% # change from sp to sf object/class
  st_shift_longitude() %>% 
  st_crop(c(xmin = st_bbox(.)[["xmin"]],
            xmax = st_bbox(.)[["xmax"]],
            ymin = -45,
            ymax = 45))

### Subsetting Planktivores
sub <- pa_rich_rls %>% filter (Troph == 'PK')
sub_size <- sizeob %>% filter (Troph == 'PK')

sub <- sub %>% left_join(sub_size, by = 'SiteCode')

### Aggregating at the regional level
sub_reg <- sub %>% group_by (Ecoregion) %>%
  summarise (SpeDens = mean (SpeDens),
             Dist_km = mean (Dist_km),
             EcoLat = mean (SiteLat),
             EcoLong = mean (SiteLong),
             Size = mean (MeanSizeTL, na.rm = T)) %>% as.data.frame

trans_long <- with(sub_reg,ifelse(EcoLong < 0, (180 - abs(EcoLong))+180, EcoLong))
sub_reg$trans_long <- trans_long

(sites <- st_as_sf(sub_reg, coords = c("trans_long", "EcoLat"), 
                   crs = 4326, agr = "constant"))


pal <-  colorRampPalette(c("#3B9AB2", "#78B7C5", "#EBCC2A", "#E1AF00","#F21A00"))

dev.off()
par(mar = c(0,0,0,0))
ggplot() +
  geom_sf(data = pacific_crop, color = '#CCCDCE', fill = '#CCCDCE', size = 0) + xlab("Longitude") + ylab("Latitude") +
  annotation_scale(location = "bl", width_hint = 0.2) +
  annotation_north_arrow(location = "bl", which_north = "true", height = unit(0.8, "cm"),width = unit(0.8, "cm"),
                         pad_x = unit(3.6, "cm"), pad_y = unit(0.01, "cm"),
                         style = north_arrow_nautical) + 
  geom_sf(data = sites,aes(size = SpeDens, fill = SpeDens), shape = 21, alpha = 0.9) +
  scale_radius(breaks=c(4,8,12,16)) +
  scale_fill_gradientn(breaks=c(4,8,12,16), colours = pal(5)) +
  coord_sf(xlim = c(20, 260), ylim = c(-45, 45), expand = FALSE) + theme_bw() +
  theme(panel.grid.major = element_line(color = gray(0.5), 
                                        size = 0.2), panel.background = element_rect(fill = "grey98"))


#----- Modelling and generating Figure 3B -----#

par(mfrow = c(2,3),mar = c(4,4,1,1))

r_sqr_reg <- vector()
coef_db_reg <- data.frame(Troph = NA, mean = NA, upp = NA, low = NA)

ord <- c('PK','OM','MI','SI','HD','GC')

for (i in 1:6){
  plot(1, type="n", xlab="", ylab="", xlim=c(900, 12500), ylim=c(0, 21), tcl = -0.3, mgp = c(3, 0.5, 0), 
       las = 1, yaxt = 'n', xaxt = 'n', bty = "n", lwd = 1.7); box(lwd = 1.4)
  axis (1, at = seq(1000, 12500, 2500), labels = seq(1000, 12500, 2500), 
        tcl = -0.3, mgp = c(3, 0.5, 0), cex.axis = 1.3, las = 1, lwd = 1.4)
  axis (2, at = seq(5,20,5), labels = seq(5,20,5), tcl = -0.3, 
        mgp = c(3, 0.5, 0), cex.axis = 1.3, las = 1, lwd = 1.4)
  
  sub <- pa_rich_rls %>% filter (Troph == ord[i])
  sub_size <- sizeob %>% filter (Troph == ord[i])
  
  sub <- sub %>% left_join(sub_size, by = 'SiteCode')
  
  ### Aggregated at the regional level ###
  sub_reg <- sub %>% group_by (Ecoregion) %>%
    summarise (SpeDens = mean (SpeDens),
               Dist_km = mean (Dist_km),
               EcoLat = mean (SiteLat),
               EcoLong = mean (SiteLong),
               Size = mean (MeanSizeTL, na.rm = T)) %>% as.data.frame
  
  mod <- glm (SpeDens + 0.1 ~ log10(Dist_km) * Size, family = Gamma (link = 'log'), 
              data = sub_reg)
  
  summ <- summary(mod)
  r_sqr_reg[i] <- with(summary(mod), 1 - deviance/null.deviance)
  lo <- loess(MeanSizeTL ~ Dist_km, data=sub, span=0.70)
  
  pred <- data.frame (Dist_km = seq (min(sub_reg$Dist_km), 12500, length = 200),
                      Size = rep(lo$fitted[1], 200))
  pred <- cbind(pred, predict(mod, pred, se.fit=TRUE, type = 'link'))
  pred <- within(pred, {
    Occ <- exp(fit)
    LL <- exp(fit - 1.96 * se.fit)
    UL <- exp(fit + 1.96 * se.fit)
  })
  
  coef_db_reg[i,] <- c(Troph = ord[i], mean = round(coef(mod)[2],3),
                       upp = round(coef(mod)[2] + 1.96 * summ$coefficients[2,2],3), 
                       low = round(coef(mod)[2] - 1.96 * summ$coefficients[2,2],3))
  
  polygon(c(pred$Dist_km,rev(pred$Dist_km)), c(pred$LL, rev(pred$UL)),
          col = alpha(colours[ord[i]],0.6), border = 'black', lwd = 0.5)
  points(sub_reg$SpeDens + 0.1 ~ sub_reg$Dist_km, pch = 21, bg = alpha(colours[ord[i]],0.9), cex = 1.5)
  lines(pred[,"Occ"] ~ pred$Dist_km, col = 'black',lwd = 1)
  text(2000,21,bquote(italic(R)^2 == .(paste0('.',strsplit(as.character(sprintf('%.2f',r_sqr_reg[i])), "\\.")[[1]][2]))), cex = 1.4)
  legend('topright',ord[i], bty = 'n', cex = 1.4)
}
mtext(side = 2, text = bquote('Species richness (spp 250'*m^-2*')'), line = -2.5, cex = 1.2, outer = T)
mtext(side = 1, text = "Distance from the Indo-Australian-Archipelago (km)", line = -2, cex = 1.2, outer = T)

###### End of the Script
